<?php

namespace App\Http\Controllers;

use App\Models\Contact;
use App\Models\Blog;
use App\Models\Listing;
use App\Models\ServiceOrder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Mail;
use Carbon\Carbon;


class AdminController extends Controller
{
    // Admin login credentials
    private $adminUsername = 'rankfetch-admin';
    private $adminPassword = 'Raf7!#$';

    // Show login page
    public function showLogin()
    {
        // If already logged in, redirect to dashboard
        if (Session::has('admin_logged_in')) {
            return redirect()->route('admin.notifications');
        }
        return view('admin.login');
    }

    // Handle login
    public function login(Request $request)
    {
        $request->validate([
            'username' => 'required|string',
            'password' => 'required|string',
        ]);

        // Check credentials
        if ($request->username === $this->adminUsername && $request->password === $this->adminPassword) {
            Session::put('admin_logged_in', true);
            Session::put('admin_username', $request->username);
            return redirect()->route('admin.notifications')->with('success', 'Login successful!');
        }

        // Error message on wrong credentials
        return redirect()->back()->withInput()->with('error', 'Wrong username or password! Please try again.');
    }

    // Handle logout
    public function logout()
    {
        Session::forget('admin_logged_in');
        Session::forget('admin_username');
        return redirect()->route('admin.login')->with('success', 'Logged out successfully!');
    }

    // Dashboard (protected by middleware)
    public function index()
    {
        $messages = Contact::latest()->get();
        $totalBlogs = Blog::count();
        $totalMessages = Contact::count();
        $totalGuestPosts = Listing::count();
        $totalOrders = ServiceOrder::count();
        $guestPostOrdersCount = ServiceOrder::where('service_type', 'Guest Post')->count();
        $serviceOrdersCount = ServiceOrder::where('service_type', '!=', 'Guest Post')->count();
        
        // Trend data for chart
        $last7Days = collect([]);
        for($i=6; $i>=0; $i--) {
            $date = Carbon::now()->subDays($i);
            $last7Days->push([
                'day' => $date->format('D'),
                'blogs' => Blog::whereDate('created_at', $date->toDateString())->count(),
                'messages' => Contact::whereDate('created_at', $date->toDateString())->count(),
                'listings' => Listing::whereDate('created_at', $date->toDateString())->count(),
                'orders' => ServiceOrder::whereDate('created_at', $date->toDateString())->count(),
            ]);
        }

        $chartLabels = $last7Days->pluck('day');
        $blogData = $last7Days->pluck('blogs');
        $messageData = $last7Days->pluck('messages');
        $listingData = $last7Days->pluck('listings');

        // Activity score including orders
        $activityData = $last7Days->map(fn($d) => $d['blogs'] + $d['messages'] + $d['listings'] + $d['orders']);

        $recentOrders = ServiceOrder::latest()->take(5)->get();

        return view('admin.dashboard', compact(
            'messages', 
            'totalBlogs', 
            'totalMessages', 
            'totalOrders', 
            'totalGuestPosts',
            'guestPostOrdersCount',
            'serviceOrdersCount',
            'chartLabels',
            'blogData',
            'messageData',
            'listingData',
            'activityData',
            'recentOrders'
        ));
    }

    public function messagesIndex()
    {
        $messages = Contact::latest()->paginate(15);
        return view('admin.messages.index', compact('messages'));
    }

    public function messagesDestroy($id)
    {
        $message = Contact::findOrFail($id);
        $message->delete();

        return redirect()->back()->with('success', 'Message deleted successfully!');
    }

    public function messagesReply(Request $request, $id)
    {
        $request->validate([
            'reply_message' => 'required|string|min:1'
        ]);

        $message = Contact::findOrFail($id);
        
        // Update status to completed first (so it happens regardless of email)
        $message->update(['status' => 'completed']);
        
        try {
            // Send reply email to user (queued for performance)
            Mail::to($message->email)->queue(new \App\Mail\AdminReplyMail($message, $request->reply_message));
            return redirect()->back()->with('success', 'Reply sent to queue and status updated! It will be delivered shortly.');
        } catch (\Exception $e) {
            \Log::error('Admin Reply Email Failed: ' . $e->getMessage());
            return redirect()->back()->with('success', 'Status updated, but email failed to queue. Check logs.');
        }
    }

    public function markAsCompleted($id)
    {
        $message = Contact::findOrFail($id);
        $message->update(['status' => 'completed']);
        return redirect()->back()->with('success', 'Message marked as completed!');
    }
}
